  /*
   *  Object %name    : %
   *  State           :  %state%
   *  Creation date   :  Wed Jan 12 12:16:55 2005
   *  Last modified   :  %modify_time%
   */
  /** @file
   *  \brief A brief description of this module
   *
   *  \version CRYS_ATP_TST_ECC_Acceptance.c#1:csrc:1
   *  \author doronp
   *  \remarks Copyright (C) 2007 by Discretix Technologies Ltd.
   *           All Rights reserved
   */
/************* Include Files ****************/ 
#include "CRYS.h"
 
#include "CRYS_ATP_TST_Defines.h"
#include "CRYS_ATP_TST_UTIL_Funcs.h"
#include "CRYS_ATP_TST_ECC_Acceptance.h"
#include "CRYS_ATP_TST_ECC_Acceptance_test_vectors.h"

#include "DX_VOS_Mem.h"
#include "math.h"
#include <stdio.h>

#if TST_USE_ECC_DATA/*In case no test vectors are selected - code disabled*/
/************************ Defines ************************/
/************************ Enums **************************/
/************************ Typedefs ***********************/
/************************ Global Data ********************/
static CRYS_ECPKI_KG_TempData_t      TST_TempData={0};
static CRYS_ECPKI_BUILD_TempData_t   TST_BuildTempData={0};
static CRYS_ECPKI_UserPrivKey_t      TST_UserPrivKey1,TST_UserPrivKey2; 
static CRYS_ECPKI_UserPublKey_t      TST_UserPublKey1,TST_UserPublKey2; 
static CRYS_ECDH_TempData_t          TST_ECDH_TempData={0}; 
static CRYS_ECPKI_UserPublKey_t        TST_UserPublKeyExported; 
static CRYS_ECDSA_SignUserContext_t    TST_SignUserContext;
static CRYS_ECDSA_VerifyUserContext_t  TST_VerifyUserContext;
static CRYS_EC_ELGAMAL_TempData_t    TST_ELGAMAL_TempData;
/************* Private function prototype ****************/
/************************ Private Functions **************/
/************************ Public Functions ***************/
/*********************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_ECC_Acceptance_ECDSA_NonIntegratedSignVerify        
 * 
 * Description:                                                                 
 *  This function Is a Acceptance test for ECC ESDSA Sign & Verify                   
 *  This function Signs & Verifies known vectors 
 * Algorithm:       
 *  1. Prepare keys 
 *     a. Generate keys 
 *     or
 *     a. Build the Private Key 
 *     b. Build the Public Key 
 *  2. Call the CRYS_ECDSA_SignInit to init the Sign operation
 *  3. Call the CRYS_ECDSA_SignUpdate - This function is called N times
 *                                        (N = number of blocks in data each block is 16byte)
 *  4. Call the CRYS_ECDSA_SignFinish to complete the sign operation
 *  5. Call the CRYS_ECDSA_VerifyInit to init the verification operation
 *  7. Call the CRYS_ECDSA_VerifyUpdate - This function is called N times
 *                                        (N = number of blocks in data each block is 16byte)
 *  7.Call the CRYS_ECDSA_VerifyFinish to complete the verification operation     
 *
 * Inputs:
 *  None
 *
 * Outputs:
 *  CRYSError_t - Function error return         
 *
 *******************************************************************************/                                          
CRYSError_t CRYS_ATP_TST_ECC_Acceptance_ECDSA_NonIntegratedSignVerify  ( int TST_KeyGeneration_TestStatus )
{
     CRYSError_t                            TST_TestStatus=0;
     int                                    TST_index,NumOfBlocks,Block_index,TST_KeyGenType;
     char       TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_ECC_Acceptance_ECDSA_NonIntegratedSignVerify"; 

     /* Setting if to use internal key generation */
     if (TST_KeyGeneration_TestStatus == TST_NO_INTERNAL_KEYGEN)
         TST_KeyGenType = TST_EXTERNAL_KEYPAIR;
     else
         TST_KeyGenType = TST_INTERNAL_KG;

     for (; TST_KeyGenType <2 ; TST_KeyGenType++) /*TST_KeyGenType = 0 Internal KeyGen // TST_KeyGenType = 1 External keys*/ 
     {
        for (TST_index =0 ; TST_index < TST_NumberOf_ECC_SignVerifyAcceptanceTests ; TST_index++) /*To run on all test vectors*/
        {
          #ifdef DX_SEP_G3_VER_ONE
            if(TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_CRYS_ECPKI_DomainID != CRYS_ECPKI_DomainID_secp256r1)
              continue;
          #endif
        
            ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************************** \n"));
            ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," %s[test 1.%d]\n",TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_Name,TST_index));  
            if (TST_KeyGenType == TST_INTERNAL_KG)
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," Internal Key Generation  \n"));
            else
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," External Keys  \n"));                        
            ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************************** \n"));

            if (TST_KeyGenType == TST_INTERNAL_KG) 
            {
                /* Generating the ECC key Pair */
                /*-----------------------------*/
                TST_ATP_Error = CRYS_ECPKI_GenKeyPair(TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_CRYS_ECPKI_DomainID,   
                                                      &TST_UserPrivKey1,
                                                      &TST_UserPublKey1,   
                                                      &TST_TempData);

                TST_ATP_CHECK_ERROR_RETURN(TST_ATP_Error,"CRYS_ECPKI_GenKeyPair",
                                                          TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_Name,
                                                          TST_MasterTestName);

            }
            else
            {
                /* Building the ECC public Key */
                /*-----------------------------*/
                TST_ATP_Error = CRYS_ECPKI_BuildPublKeyFullCheck(
                                              TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_CRYS_ECPKI_DomainID,      
                                                        TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_ExternalPublKey,
                                                        TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_ExternalPublKeySize,
                                                        &TST_UserPublKey1,
                            &TST_BuildTempData); 

                TST_ATP_CHECK_ERROR_RETURN(TST_ATP_Error,"CRYS_ECPKI_BuildPublKey",
                                                          TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_Name,
                                                          TST_MasterTestName);


                /* Building the ECC private Key */
                /*------------------------------*/
                TST_ATP_Error = CRYS_ECPKI_BuildPrivKey(
                                              TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_CRYS_ECPKI_DomainID,   
                                                        TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_ExternalPrivKey,
                                                        TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_ExternalPrivKeySize,
                                                        &TST_UserPrivKey1);

                TST_ATP_CHECK_ERROR_RETURN(TST_ATP_Error,"CRYS_ECPKI_BuildPrivKey",
                                                          TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_Name,
                                                          TST_MasterTestName);

            }

            /* Initializing the Sign operation */
            /*---------------------------------*/
            TST_ATP_Error = CRYS_ECDSA_SignInit(&TST_SignUserContext,
                                                &TST_UserPrivKey1,
                                                TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_HashMode);

            TST_ATP_CHECK_ERROR_RETURN(TST_ATP_Error,"CRYS_ECDSA_SignInit",
                                                    TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_Name,
                                                    TST_MasterTestName);



            /* Sign the data using the private Key */
            /* Calling the update function         */
            /*-------------------------------------*/
            NumOfBlocks = TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_input_dataSize/TST_MAX_SINGLE_ECC_SIGN_BLOCK_SIZE_IN_BYTS;
             
            /* Debug massages */
            ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"number of blocks = %1d \n",NumOfBlocks));
            ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"Input Data Size [byte] = %1d \n" ,TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_input_dataSize));
            ATP_LOG_TST_PRINT((CRYS_ATP_TST_DBG_HIGH,"SingleBlockSize [byte] = %1d \n",TST_MAX_SINGLE_ECC_SIGN_BLOCK_SIZE_IN_BYTS));
             
             
            for (Block_index = 0 ; Block_index < NumOfBlocks ; Block_index++)/* ECDSA SignUpdate block loop*/
            { 
                TST_ATP_Error = CRYS_ECDSA_SignUpdate(&TST_SignUserContext,
                                                    TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_input_data+(TST_MAX_SINGLE_ECC_SIGN_BLOCK_SIZE_IN_BYTS*Block_index),
                                                    TST_MAX_SINGLE_ECC_SIGN_BLOCK_SIZE_IN_BYTS*sizeof(DxUint8_t));                                        
                                         
                TST_ATP_CHECK_ERROR_RETURN(TST_ATP_Error,"CRYS_ECDSA_SignUpdate",
                                                        TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_Name,
                                                        TST_MasterTestName);
                                    
            }                       

            /* Finish the Sign operation */
            /*---------------------------*/
            TST_ATP_Error = CRYS_ECDSA_SignFinish(&TST_SignUserContext,
                                                TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_output_signiture,
                            &TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_SignitureSize);
                     
            TST_ATP_CHECK_ERROR_RETURN(TST_ATP_Error,"CRYS_ECDSA_SignFinish",
                                                    TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_Name,
                                                    TST_MasterTestName);


            /* Verify the data using the public Key */
            /* Initializing the Verify operation    */
            /*--------------------------------------*/
            TST_ATP_Error = CRYS_ECDSA_VerifyInit(&TST_VerifyUserContext,
                                                &TST_UserPublKey1,
                                                TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_HashMode);

            TST_ATP_CHECK_ERROR_RETURN(TST_ATP_Error,"CRYS_ECDSA_VerifyInit",
                                                    TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_Name,
                                                    TST_MasterTestName);



            /* Calling the update function */
            /*-----------------------------*/
            for (Block_index = 0 ; Block_index < NumOfBlocks ; Block_index++)/* ECDSA VerifyUpdate block loop*/
            { 
                TST_ATP_Error = CRYS_ECDSA_VerifyUpdate(&TST_VerifyUserContext,
                                                        TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_input_data+(TST_MAX_SINGLE_ECC_SIGN_BLOCK_SIZE_IN_BYTS*Block_index),
                                                        TST_MAX_SINGLE_ECC_SIGN_BLOCK_SIZE_IN_BYTS*sizeof(DxUint8_t));                                        
                                         
                TST_ATP_CHECK_ERROR_RETURN(TST_ATP_Error,"CRYS_ECDSA_VerifyUpdate",
                                                          TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_Name,
                                                          TST_MasterTestName);
                                     
            } 

            /* Finish the Verify operation */
            /*-----------------------------*/
            TST_ATP_Error = CRYS_ECDSA_VerifyFinish(&TST_VerifyUserContext,
                                                    TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_output_signiture,
                                TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_SignitureSize);
                     
            TST_ATP_CHECK_ERROR_RETURN(TST_ATP_Error,"CRYS_ECDSA_VerifyFinish",
                                                    TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_Name,
                                                    TST_MasterTestName);


        }/*End of TST_Index loop*/
     }/* End of TST_KeyGenType loop*/
     return TST_TestStatus;
     
EXIT_ON_ERROR:
  {
   return DX_FAILURE;
  }
}/*End of CRYS_ATP_TST_ECC_Acceptance_ECDSA_NonIntegratedSignVerify*/
/***************************************************************************************************************************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***************************************************************************************************************************************************************/
/*********************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_ECC_Acceptance_ECDSA_IntegratedSignVerify        
 * 
 * Inputs:
 *  None
 *
 * Outputs:
 *  CRYSError_t - Function error return                                     
 *                                
 *
 * Description:                                                                 
 *  This function Is a Acceptance test for ECC ESDSA Sign & Verify                   
 *  This function Signs & Verifies known vectors 
 * Algorithm:       
 *  1. Prepare keys 
 *     a. Generate keys 
 *     or
 *     a. Build the Private Key 
 *     b. Build the Public Key 
 *  2. Call CRYS_ECDSA_Sign to perform the sign operation
 *  3. Call CRYS_ECDSA_Verify to perform the verification operation   
 *
 ******************************************************************************/                                          
CRYSError_t CRYS_ATP_TST_ECC_Acceptance_ECDSA_IntegratedSignVerify  ( int TST_KeyGeneration_TestStatus )
{
     CRYSError_t                            TST_TestStatus=0;
     int                                    TST_index = 0,TST_KeyGenType;
     char       TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_ECC_Acceptance_ECDSA_IntegratedSignVerify"; 

     /* Setting if to use internal key generation */
     if (TST_KeyGeneration_TestStatus == TST_NO_INTERNAL_KEYGEN)
         TST_KeyGenType = TST_EXTERNAL_KEYPAIR;
     else
         TST_KeyGenType = TST_INTERNAL_KG;

     for (; TST_KeyGenType <2 ; TST_KeyGenType++) /*TST_KeyGenType = 0 Internal KeyGen // TST_KeyGenType = 1 External keys*/ 
     {
        for (TST_index =0 ; TST_index < TST_NumberOf_ECC_SignVerifyAcceptanceTests ; TST_index++) /*To run on all test vectors*/
        {
          #ifdef DX_SEP_G3_VER_ONE
            if(TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_CRYS_ECPKI_DomainID != CRYS_ECPKI_DomainID_secp256r1)
              continue;
          #endif
            ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************************** \n"));
            ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," %s[test 1.%d]\n",TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_Name,TST_index));  
            if (TST_KeyGenType == TST_INTERNAL_KG)
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," Internal Key Generation  \n"));
            else
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," External Keys  \n"));                        
            ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************************** \n"));

            if (TST_KeyGenType == TST_INTERNAL_KG) 
            {
                /* Generating the ECC key Pair */
                /*-----------------------------*/
                TST_ATP_Error = CRYS_ECPKI_GenKeyPair(TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_CRYS_ECPKI_DomainID,   
                                                      &TST_UserPrivKey1,
                                                      &TST_UserPublKey1,   
                                                      &TST_TempData);

                TST_ATP_CHECK_ERROR_RETURN(TST_ATP_Error,"CRYS_ECPKI_GenKeyPair",
                                                          TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_Name,
                                                          TST_MasterTestName);

            }
            else
            {
                /* Building the ECC public Key */
                /*-----------------------------*/
                TST_ATP_Error = CRYS_ECPKI_BuildPublKeyFullCheck(
                                              TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_CRYS_ECPKI_DomainID,      
                                                        TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_ExternalPublKey,
                                                        TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_ExternalPublKeySize,
                            &TST_UserPublKey1,
                            &TST_BuildTempData); 

                TST_ATP_CHECK_ERROR_RETURN(TST_ATP_Error,"CRYS_ECPKI_BuildPublKey",
                                                          TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_Name,
                                                          TST_MasterTestName);


                /* Building the ECC private Key */
                /*------------------------------*/
                TST_ATP_Error = CRYS_ECPKI_BuildPrivKey(TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_CRYS_ECPKI_DomainID,   
                                                        TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_ExternalPrivKey,
                                                        TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_ExternalPrivKeySize,
                                                        &TST_UserPrivKey1);

                TST_ATP_CHECK_ERROR_RETURN(TST_ATP_Error,"CRYS_ECPKI_BuildPrivKey",
                                                          TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_Name,
                                                          TST_MasterTestName);

            }

            /* Performing the Sign operation */
            /*-------------------------------*/
            TST_ATP_Error = CRYS_ECDSA_Sign(&TST_SignUserContext,
                                  &TST_UserPrivKey1,
                                  TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_HashMode,                
                                  TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_input_data,      
                                  TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_input_dataSize,
                                            TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_output_signiture,
                                            &TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_SignitureSize);

            TST_ATP_CHECK_ERROR_RETURN(TST_ATP_Error,"CRYS_ECDSA_SignIntegrated",
                                                    TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_Name,
                                                    TST_MasterTestName);



            /* Verify the data using the public Key */
            /*--------------------------------------*/
            TST_ATP_Error = CRYS_ECDSA_Verify(&TST_VerifyUserContext,
                                            &TST_UserPublKey1,
                                            TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_HashMode,
                                            TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_output_signiture,
                                            TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_SignitureSize,
                                            TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_input_data, 
                                            TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_input_dataSize);

            TST_ATP_CHECK_ERROR_RETURN(TST_ATP_Error,"CRYS_ECDSA_VerifyIntegrated",
                                                    TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_Name,
                                                    TST_MasterTestName);


         }/*End of TST_Index loop*/
     }/* End of TST_KeyGenType loop*/
     return TST_TestStatus;
     
EXIT_ON_ERROR:
  {
   return DX_FAILURE;
  }
}/*End of CRYS_ATP_TST_ECC_Acceptance_ECDSA_IntegratedSignVerify*/
/***************************************************************************************************************************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***************************************************************************************************************************************************************/
/*********************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_ECC_Acceptance_DH        
 * 
 * Inputs:
 *  None
 *
 * Outputs:
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                      
 * To test the Diffie-Hellman key shared secret generation using ECC capabilities
 * Algorithm: 
 * 1.Build or generate the private and public keys for user 1 using       
 *     a. Generate keys 
 *     or
 *     a. Build the Private Key 
 *     b. Build the Public Key 
 * 2.Build or generate the private and public keys for user 2 using 
 *     a. Generate keys 
 *     or
 *     a. Build the Private Key 
 *     b. Build the Public Key 
 * 3. Call CRYS_ECDH_SVDP_DH  to Create a shared secret 1 for user 1 
 *    using user1 private key and user 2 public key 
 * 4. Call CRYS_ECDH_SVDP_DH  to Create a shared secret 2 for user 2 
 *    using user2 private key and user 1 public key
 * 5. Compare the two secret keys - to verify they are identical
 *
 ******************************************************************************/                                          
CRYSError_t CRYS_ATP_TST_ECC_Acceptance_DH  ( int TST_KeyGeneration_TestStatus)
{
     CRYSError_t                            TST_TestStatus=0;
     int                                    TST_index = 0,TST_KeyGenType;
     DxUint8_t                              TST_ShareSecret1[TST_DH_MAX_SECRET_LEN_BYTE];
     DxUint8_t                              TST_ShareSecret2[TST_DH_MAX_SECRET_LEN_BYTE];
     DxUint32_t                             TST_ShareSecret1Size,TST_ShareSecret2Size;
     char       TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_ECC_Acceptance_DH"; 

     /* Setting if to use internal key generation */
     if (TST_KeyGeneration_TestStatus == TST_NO_INTERNAL_KEYGEN)
         TST_KeyGenType = TST_EXTERNAL_KEYPAIR;
     else
         TST_KeyGenType = TST_INTERNAL_KG;
     for (; TST_KeyGenType <2 ; TST_KeyGenType++) /*TST_KeyGenType = 0 Internal KeyGen // TST_KeyGenType = 1 External keys*/ 
     {
        for (TST_index =0 ; TST_index < TST_NumberOf_ECC_DHAcceptanceTests ; TST_index++) /*To run on all test vectors*/
        {
          #ifdef DX_SEP_G3_VER_ONE
            if(TST_ECC_Acceptance_DH_TestData[TST_index].TST_CRYS_ECPKI_DomainID != CRYS_ECPKI_DomainID_secp256r1)
              continue;
          #endif
            /*Test Data initialization */
            /*-------------------------*/
            TST_ShareSecret1Size = sizeof(TST_ShareSecret1);
            TST_ShareSecret2Size = sizeof(TST_ShareSecret2);

            ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************************** \n"));
            ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," %s[test 1.%d]\n",TST_ECC_Acceptance_DH_TestData[TST_index].TST_Name,TST_index));  
            if (TST_KeyGenType == TST_INTERNAL_KG)
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," Internal Key Generation  \n"));
            else
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," External Keys  \n"));                        
            ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************************** \n"));

            if (TST_KeyGenType == TST_INTERNAL_KG) 
            {
                /* Generating the ECC key Pair for user 1*/
                /*---------------------------------------*/
                TST_ATP_Error = CRYS_ECPKI_GenKeyPair(TST_ECC_Acceptance_DH_TestData[TST_index].TST_CRYS_ECPKI_DomainID,   
                                                    &TST_UserPrivKey1,
                                                    &TST_UserPublKey1,   
                                                    &TST_TempData);

                TST_ATP_CHECK_ERROR_RETURN(TST_ATP_Error,"CRYS_ECPKI_GenKeyPair -  User 1",
                                                        TST_ECC_Acceptance_DH_TestData[TST_index].TST_Name,
                                                        TST_MasterTestName);


                /* Generating the ECC key Pair for user 2*/
                /*---------------------------------------*/
                TST_ATP_Error = CRYS_ECPKI_GenKeyPair(TST_ECC_Acceptance_DH_TestData[TST_index].TST_CRYS_ECPKI_DomainID,   
                                                    &TST_UserPrivKey2,
                                                    &TST_UserPublKey2,   
                                                    &TST_TempData);

                TST_ATP_CHECK_ERROR_RETURN(TST_ATP_Error,"CRYS_ECPKI_GenKeyPair - User 2",
                                                        TST_ECC_Acceptance_DH_TestData[TST_index].TST_Name,
                                                        TST_MasterTestName);

            }
            else
            {
                /* Building the ECC public Key for user 1 */
                /*-----------------------------------------*/
                TST_ATP_Error = CRYS_ECPKI_BuildPublKeyFullCheck(
                                              TST_ECC_Acceptance_DH_TestData[TST_index].TST_CRYS_ECPKI_DomainID,      
                                                        TST_ECC_Acceptance_DH_TestData[TST_index].TST_ExternalPublKey1,
                                                        TST_ECC_Acceptance_DH_TestData[TST_index].TST_ExternalPublKey1Size,
                            &TST_UserPublKey1,
                            &TST_BuildTempData ); 

                TST_ATP_CHECK_ERROR_RETURN(TST_ATP_Error,"CRYS_ECPKI_BuildPublKey - User 1",
                                                          TST_ECC_Acceptance_DH_TestData[TST_index].TST_Name,
                                                          TST_MasterTestName);


                /* Building the ECC public Key for user 2 */
                /*-----------------------------------------*/
                TST_ATP_Error = CRYS_ECPKI_BuildPublKeyFullCheck(TST_ECC_Acceptance_DH_TestData[TST_index].TST_CRYS_ECPKI_DomainID,      
                                                        TST_ECC_Acceptance_DH_TestData[TST_index].TST_ExternalPublKey2,
                                                        TST_ECC_Acceptance_DH_TestData[TST_index].TST_ExternalPublKey2Size,
                             &TST_UserPublKey2,
                            &TST_BuildTempData ); 

                TST_ATP_CHECK_ERROR_RETURN(TST_ATP_Error,"CRYS_ECPKI_BuildPublKey - User 2",
                                                          TST_ECC_Acceptance_DH_TestData[TST_index].TST_Name,
                                                          TST_MasterTestName);


                /* Building the ECC private Key for user 1 */
                /*-----------------------------------------*/
                TST_ATP_Error = CRYS_ECPKI_BuildPrivKey(TST_ECC_Acceptance_DH_TestData[TST_index].TST_CRYS_ECPKI_DomainID,   
                                                        TST_ECC_Acceptance_DH_TestData[TST_index].TST_ExternalPrivKey1,
                                                        TST_ECC_Acceptance_DH_TestData[TST_index].TST_ExternalPrivKey1Size,
                                                        &TST_UserPrivKey1);

                TST_ATP_CHECK_ERROR_RETURN(TST_ATP_Error,"CRYS_ECPKI_BuildPrivKey - User 1",
                                                          TST_ECC_Acceptance_DH_TestData[TST_index].TST_Name,
                                                          TST_MasterTestName);


                /* Building the ECC private Key for user 2 */
                /*-----------------------------------------*/
                TST_ATP_Error = CRYS_ECPKI_BuildPrivKey(TST_ECC_Acceptance_DH_TestData[TST_index].TST_CRYS_ECPKI_DomainID,   
                                                        TST_ECC_Acceptance_DH_TestData[TST_index].TST_ExternalPrivKey2,
                                                        TST_ECC_Acceptance_DH_TestData[TST_index].TST_ExternalPrivKey2Size,
                                                        &TST_UserPrivKey2);

                TST_ATP_CHECK_ERROR_RETURN(TST_ATP_Error,"CRYS_ECPKI_BuildPrivKey - User 2",
                                                          TST_ECC_Acceptance_DH_TestData[TST_index].TST_Name,
                                                          TST_MasterTestName);

            }

            /* Generating the Secret for user 1*/
            /*---------------------------------*/
            TST_ATP_Error = CRYS_ECDH_SVDP_DH(&TST_UserPublKey2,
                                              &TST_UserPrivKey1,
                                  TST_ShareSecret1,
                                  &TST_ShareSecret1Size,
                                  &TST_ECDH_TempData);

            TST_ATP_CHECK_ERROR_RETURN(TST_ATP_Error,"CRYS_ECDH_SVDP_DH - User 1",
                                                    TST_ECC_Acceptance_DH_TestData[TST_index].TST_Name,
                                                    TST_MasterTestName);


            /* Generating the Secret for user 2*/
            /*---------------------------------*/
            TST_ATP_Error = CRYS_ECDH_SVDP_DH(&TST_UserPublKey1,
                                            &TST_UserPrivKey2,
                                TST_ShareSecret2,
                                &TST_ShareSecret2Size,
                                &TST_ECDH_TempData);

            TST_ATP_CHECK_ERROR_RETURN(TST_ATP_Error,"CRYS_ECDH_SVDP_DH - User 2",
                                                    TST_ECC_Acceptance_DH_TestData[TST_index].TST_Name,
                                                    TST_MasterTestName);


            /* Verifying we have the same Secret Key both for client & Server */
            /*----------------------------------------------------------------*/                                                    
            TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp(TST_ShareSecret1 ,
                                                    TST_ShareSecret2,
                                                    TST_ShareSecret2Size);   
                                    
            ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                        "\nShared Secret 1--\n",
                                        TST_ShareSecret1,
                                        TST_ShareSecret1Size);   
                                        
            ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                        "\nShared Secret 2--\n",
                                        TST_ShareSecret2,
                                        TST_ShareSecret2Size);                                                                                   

            TST_ATP_CHECK_ERROR_RETURN(TST_ATP_Error,
                                                    "DX_VOS_MemCmp - shared secret", 
                                                    TST_ECC_Acceptance_DH_TestData[TST_index].TST_Name,
                                                    TST_MasterTestName);
  
         }/*End of TST_Index loop*/
     }/* End of TST_KeyGenType loop*/
     return TST_TestStatus;
    
EXIT_ON_ERROR:
  {
   return DX_FAILURE;
  }

}/*End of CRYS_ATP_TST_ECC_Acceptance_DH*/
/***************************************************************************************************************************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***************************************************************************************************************************************************************/
/*********************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_ECC_Acceptance_ElGamal_EncDec        
 * 
 * Inputs:
 *  None
 *
 * Outputs:
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                      
 * To test the El-Gamal algorithm for encryption and decryption
 *
 * Algorithm:      
 *  1. Prepare keys 
 *     a. Generate keys 
 *     or
 *     a. Build the Private Key 
 *     b. Build the Public Key 
 * 2. Encrypt a block of data using the CRYS_ECPKI_ELGAMAL_Encrypt encryption process
 * 3. Decrypt a block of data using CRYS_ECPKI_ELGAMAL_Decrypt decryption process 
 * 4. Compare the decrypted data to the original data
 *
 ******************************************************************************/                                          
CRYSError_t CRYS_ATP_TST_ECC_Acceptance_ElGamal_EncDec  ( int TST_KeyGeneration_TestStatus)
{
     CRYSError_t                            TST_TestStatus=0;
     int                                    TST_index = 0,TST_KeyGenType;
     char       TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_ECC_Acceptance_ElGamal_EncDec"; 

     /* Setting if to use internal key generation */
     if (TST_KeyGeneration_TestStatus == TST_NO_INTERNAL_KEYGEN)
         TST_KeyGenType = TST_EXTERNAL_KEYPAIR;
     else
         TST_KeyGenType = TST_INTERNAL_KG;

     for (; TST_KeyGenType <2 ; TST_KeyGenType++) /*TST_KeyGenType = 0 Internal KeyGen // TST_KeyGenType = 1 External keys*/ 
     {
        for (TST_index =0 ; TST_index < TST_NumberOf_ECC_ElGamalAcceptanceTests ; TST_index++) /*To run on all test vectors*/
        {
          #ifdef DX_SEP_G3_VER_ONE
            if(TST_ECC_Acceptance_ElGamal_EncDec_TestData[TST_index].TST_CRYS_ECPKI_DomainID != CRYS_ECPKI_DomainID_secp256r1)
              continue;
          #endif

      /* ECC ELGAMAL functions should work only with the WMDRM domain */
      if (TST_ECC_Acceptance_ElGamal_EncDec_TestData[TST_index].TST_CRYS_ECPKI_DomainID != CRYS_ECPKI_DomainID_WMDRM10)
        continue;

            ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************************** \n"));
            ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," %s[test 1.%d]\n",TST_ECC_Acceptance_ElGamal_EncDec_TestData[TST_index].TST_Name,TST_index));  
            if (TST_KeyGenType == TST_INTERNAL_KG)
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," Internal Key Generation  \n"));
            else
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," External Keys  \n"));                        
            ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************************** \n"));

            if (TST_KeyGenType == TST_INTERNAL_KG) 
            {
                /* Generating the ECC key Pair */
                /*-----------------------------*/
                TST_ATP_Error = CRYS_ECPKI_GenKeyPair(TST_ECC_Acceptance_ElGamal_EncDec_TestData[TST_index].TST_CRYS_ECPKI_DomainID,   
                                                      &TST_UserPrivKey1,
                                                      &TST_UserPublKey1,   
                                                      &TST_TempData);

                TST_ATP_CHECK_ERROR_RETURN(TST_ATP_Error,"CRYS_ECPKI_GenKeyPair",
                                                          TST_ECC_Acceptance_ElGamal_EncDec_TestData[TST_index].TST_Name,
                                                          TST_MasterTestName);

            }
            else
            {
                /* Building the ECC public Key */
                /*-----------------------------*/
                TST_ATP_Error = CRYS_ECPKI_BuildPublKeyFullCheck(TST_ECC_Acceptance_ElGamal_EncDec_TestData[TST_index].TST_CRYS_ECPKI_DomainID,      
                                                        TST_ECC_Acceptance_ElGamal_EncDec_TestData[TST_index].TST_ExternalPublKey,
                                                        TST_ECC_Acceptance_ElGamal_EncDec_TestData[TST_index].TST_ExternalPublKeySize,
                             &TST_UserPublKey1,
                            &TST_BuildTempData ); 

                TST_ATP_CHECK_ERROR_RETURN(TST_ATP_Error,"CRYS_ECPKI_BuildPublKey",
                                                          TST_ECC_Acceptance_ElGamal_EncDec_TestData[TST_index].TST_Name,
                                                          TST_MasterTestName);


                /* Building the ECC private Key */
                /*------------------------------*/
                TST_ATP_Error = CRYS_ECPKI_BuildPrivKey(TST_ECC_Acceptance_ElGamal_EncDec_TestData[TST_index].TST_CRYS_ECPKI_DomainID,   
                                                        TST_ECC_Acceptance_ElGamal_EncDec_TestData[TST_index].TST_ExternalPrivKey,
                                                        TST_ECC_Acceptance_ElGamal_EncDec_TestData[TST_index].TST_ExternalPrivKeySize,
                                                        &TST_UserPrivKey1);

                TST_ATP_CHECK_ERROR_RETURN(TST_ATP_Error,"CRYS_ECPKI_BuildPrivKey",
                                                          TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_Name,
                                                          TST_MasterTestName);

            }
            /* Encrypting the massage */
            /*------------------------*/
            TST_ATP_Error =  CRYS_ECPKI_ELGAMAL_Encrypt(&TST_UserPublKey1,
                                                        TST_ECC_Acceptance_ElGamal_EncDec_TestData[TST_index].TST_input_data,
                                                        TST_ECC_Acceptance_ElGamal_EncDec_TestData[TST_index].TST_input_dataSize,
                                                        TST_ECC_Acceptance_ElGamal_EncDec_TestData[TST_index].TST_output_data,
                                                        &TST_ECC_Acceptance_ElGamal_EncDec_TestData[TST_index].TST_output_dataSize,
                                                        &TST_ELGAMAL_TempData);

            TST_ATP_CHECK_ERROR_RETURN(TST_ATP_Error,"CRYS_ECPKI_ELGAMAL_Encrypt",
                                                    TST_ECC_Acceptance_ElGamal_EncDec_TestData[TST_index].TST_Name,
                                                    TST_MasterTestName);


            /* Decrypting the massage */
            /*------------------------*/
            TST_ATP_Error =  CRYS_ECPKI_ELGAMAL_Decrypt(&TST_UserPrivKey1,
                                                        TST_ECC_Acceptance_ElGamal_EncDec_TestData[TST_index].TST_output_data,
                                                        TST_ECC_Acceptance_ElGamal_EncDec_TestData[TST_index].TST_output_dataSize,
                                                        TST_ECC_Acceptance_ElGamal_EncDec_TestData[TST_index].TST_Decrypted_data,
                                                        &TST_ECC_Acceptance_ElGamal_EncDec_TestData[TST_index].TST_Decrypted_dataSize,
                                                        &TST_ELGAMAL_TempData);

            TST_ATP_CHECK_ERROR_RETURN(TST_ATP_Error,"CRYS_ECPKI_ELGAMAL_Decrypt",
                                                    TST_ECC_Acceptance_ElGamal_EncDec_TestData[TST_index].TST_Name,
                                                    TST_MasterTestName);


            /* Comparing the decrypted data to the original  */
            /*-----------------------------------------------*/                                                    
            TST_ATP_Error = (CRYSError_t)DX_VOS_MemCmp(TST_ECC_Acceptance_ElGamal_EncDec_TestData[TST_index].TST_Decrypted_data,
                                                       TST_ECC_Acceptance_ElGamal_EncDec_TestData[TST_index].TST_input_data,
                                                       TST_ECC_Acceptance_ElGamal_EncDec_TestData[TST_index].TST_input_dataSize);   
                                    
            ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                        "\n Decrypted data--\n",
                                        TST_ECC_Acceptance_ElGamal_EncDec_TestData[TST_index].TST_Decrypted_data,
                                        TST_ECC_Acceptance_ElGamal_EncDec_TestData[TST_index].TST_Decrypted_dataSize);   
                                        
            ATP_LOG_TST_DisplayBuffer (CRYS_ATP_TST_DBG_HIGH,
                                        "\n input data--\n",
                                        TST_ECC_Acceptance_ElGamal_EncDec_TestData[TST_index].TST_input_data,
                                        TST_ECC_Acceptance_ElGamal_EncDec_TestData[TST_index].TST_input_dataSize);                                                                                   

            TST_ATP_CHECK_ERROR_RETURN(TST_ATP_Error,
                                                    "DX_VOS_MemCmp - decryption", 
                                                    TST_ECC_Acceptance_ElGamal_EncDec_TestData[TST_index].TST_Name,
                                                    TST_MasterTestName);

       
         }/*End of TST_Index loop*/
     }/* End of TST_KeyGenType loop*/
     return TST_TestStatus;
     
EXIT_ON_ERROR:
  {
   return DX_FAILURE;
  }
}/*End of CRYS_ATP_TST_ECC_Acceptance_ElGamal_EncDec*/
/***************************************************************************************************************************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***************************************************************************************************************************************************************/
/*********************************************************
 * Function Name:                                                           
 *  CRYS_ATP_TST_ECC_Acceptance_ImportExportKeyPair        
 * 
 * Inputs:
 *  None
 *
 * Outputs:
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                      
 * To test the ECC export public key functionality
 *
 * Algorithm:      
 *  1. Prepare keys 
 *     a. Generate keys 
 *     or
 *     a. Build the Private Key 
 *     b. Build the Public Key 
 *  2.  Export the public key using CRYS_ECPKI_ExportPublKey
 *  3.  Re-build the exported key using CRYS_ECPKI_Build_PubKey
 *  4.  Sign a massage using the exported public key using CRYS_ECDSA_Sign to perform the sign operation
 *  5.  Verify the massage using the appropriate public key using CRYS_ECDSA_Verify
 *      to perform the verification operation
 *
 ******************************************************************************/                                          
CRYSError_t CRYS_ATP_TST_ECC_Acceptance_ImportExportKeyPair  ( int TST_KeyGeneration_TestStatus)
{
     CRYSError_t                                       TST_TestStatus=0;    
     DxUint8_t  TST_UserPublKeyExported_buff[TST_MAX_ECC_KEY_SIZE_BYTES]; 
     DxUint32_t TST_UserPublKeyExported_buffSize;
     int        TST_index = 0,TST_KeyGenType;
     char       TST_MasterTestName[TST_FUNC_STRING_LEN] = "CRYS_ATP_TST_ECC_Acceptance_ImportExportKeyPair"; 

     /* Setting if to use internal key generation */
     if (TST_KeyGeneration_TestStatus == TST_NO_INTERNAL_KEYGEN)
         TST_KeyGenType = TST_EXTERNAL_KEYPAIR;
     else
         TST_KeyGenType = TST_INTERNAL_KG;

     for (; TST_KeyGenType <2 ; TST_KeyGenType++) /*TST_KeyGenType = 0 Internal KeyGen // TST_KeyGenType = 1 External keys*/ 
     {
        for (TST_index =0 ; TST_index < TST_NumberOf_ECC_SignVerifyAcceptanceTests ; TST_index++) /*To run on all test vectors*/
        {
          #ifdef DX_SEP_G3_VER_ONE
            if(TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_CRYS_ECPKI_DomainID != CRYS_ECPKI_DomainID_secp256r1)
              continue;
          #endif
            ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************************** \n"));
            ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," %s[test 1.%d]\n",TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_Name,TST_index));  
            if (TST_KeyGenType == TST_INTERNAL_KG)
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," Internal Key Generation  \n"));
            else
                ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG," External Keys  \n"));                        
            ATP_LOG_TST_PRINT((CRYS_ATP_TST_MSG,"********************************************************************** \n"));

            /*Initializing the data*/
            TST_UserPublKeyExported_buffSize = TST_MAX_ECC_KEY_SIZE_BYTES;

            if (TST_KeyGenType == TST_INTERNAL_KG) 
            {
                /* Generating the ECC key Pair */
                /*-----------------------------*/
                TST_ATP_Error = CRYS_ECPKI_GenKeyPair(TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_CRYS_ECPKI_DomainID,   
                                                      &TST_UserPrivKey1,
                                                      &TST_UserPublKey1,   
                                                      &TST_TempData);

                TST_ATP_CHECK_ERROR_RETURN(TST_ATP_Error,"CRYS_ECPKI_GenKeyPair",
                                                          TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_Name,
                                                          TST_MasterTestName);

            }
            else
            {
                /* Building the ECC public Key */
                /*-----------------------------*/
                TST_ATP_Error = CRYS_ECPKI_BuildPublKeyFullCheck(
                                              TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_CRYS_ECPKI_DomainID,      
                                                        TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_ExternalPublKey,
                                                        TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_ExternalPublKeySize,
                             &TST_UserPublKey1,
                            &TST_BuildTempData ); 

                TST_ATP_CHECK_ERROR_RETURN(TST_ATP_Error,"CRYS_ECPKI_BuildPublKey",
                                                          TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_Name,
                                                          TST_MasterTestName);


                /* Building the ECC private Key */
                /*------------------------------*/
                TST_ATP_Error = CRYS_ECPKI_BuildPrivKey(TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_CRYS_ECPKI_DomainID,   
                                                        TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_ExternalPrivKey,
                                                        TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_ExternalPrivKeySize,
                                                        &TST_UserPrivKey1);

                TST_ATP_CHECK_ERROR_RETURN(TST_ATP_Error,"CRYS_ECPKI_BuildPrivKey",
                                                          TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_Name,
                                                          TST_MasterTestName);

            }
            /* Export the public key the massage */
            /*-----------------------------------*/
            TST_ATP_Error =  CRYS_ECPKI_ExportPublKey(&TST_UserPublKey1,
                                                      CRYS_EC_PointUncompressed,
                                                      TST_UserPublKeyExported_buff,
                                                      &TST_UserPublKeyExported_buffSize); 

            TST_ATP_CHECK_ERROR_RETURN(TST_ATP_Error,"CRYS_ECPKI_ExportPublKey",
                                                    TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_Name,
                                                    TST_MasterTestName);


            /*Import the exported key to a different pubkey*/
            /*---------------------------------------------*/
            TST_ATP_Error = CRYS_ECPKI_BuildPublKeyFullCheck(
                                            TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_CRYS_ECPKI_DomainID,      
                                                    TST_UserPublKeyExported_buff,
                                                    TST_UserPublKeyExported_buffSize,
                           &TST_UserPublKeyExported,
                          &TST_BuildTempData ); 

            TST_ATP_CHECK_ERROR_RETURN(TST_ATP_Error,"CRYS_ECPKI_BuildPublKey - From exported",
                                                      TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_Name,
                                                      TST_MasterTestName);

            /* Performing the Sign operation */
            /*-------------------------------*/
            TST_ATP_Error = CRYS_ECDSA_Sign(&TST_SignUserContext,
                                  &TST_UserPrivKey1,
                                  TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_HashMode,                
                                  TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_input_data,      
                                  TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_input_dataSize,
                                            TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_output_signiture,
                                            &TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_SignitureSize);

            TST_ATP_CHECK_ERROR_RETURN(TST_ATP_Error,"CRYS_ECDSA_Sign",
                                                    TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_Name,
                                                    TST_MasterTestName);


            /* Verify the data using the exported public Key */
            /*-----------------------------------------------*/
            TST_ATP_Error = CRYS_ECDSA_Verify(&TST_VerifyUserContext,
                                              &TST_UserPublKeyExported,
                                              TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_HashMode,
                                              TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_output_signiture,
                                              TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_SignitureSize,
                                              TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_input_data, 
                                              TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_input_dataSize);

            TST_ATP_CHECK_ERROR_RETURN(TST_ATP_Error,"CRYS_ECDSA_Verify with exporeted key",
                                                    TST_ECC_Acceptance_ECDSA_SignVerify_TestData[TST_index].TST_Name,
                                                    TST_MasterTestName);

          }/*End of TST_Index loop*/
     }/* End of TST_KeyGenType loop*/
     return TST_TestStatus;
     
EXIT_ON_ERROR:
  {
   return DX_FAILURE;
  }

}/*End of CRYS_ATP_TST_ECC_Acceptance_ImportExportKeyPair*/

#endif


